import Image from "next/image"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { ExternalLink } from "lucide-react"

interface Provider {
  logo_path: string
  provider_name: string
  provider_id: number
}

interface WhereToWatchTVProps {
  providers: {
    [country: string]: {
      link?: string
      flatrate?: Provider[]
      rent?: Provider[]
      buy?: Provider[]
    }
  }
}

export function WhereToWatchTV({ providers }: WhereToWatchTVProps) {
  const usProviders = providers.US || {}
  const { flatrate = [], rent = [], buy = [], link } = usProviders

  const hasProviders = flatrate.length > 0 || rent.length > 0 || buy.length > 0

  if (!hasProviders) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Where to Watch</CardTitle>
        </CardHeader>
        <CardContent>
          <p className="text-muted-foreground">Streaming information not available</p>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          Where to Watch
          {link && (
            <a href={link} target="_blank" rel="noopener noreferrer" className="text-primary hover:text-primary/80">
              <ExternalLink className="h-4 w-4" />
            </a>
          )}
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {flatrate.length > 0 && (
          <div>
            <Badge variant="secondary" className="mb-3">
              Stream
            </Badge>
            <div className="grid grid-cols-3 gap-3">
              {flatrate.map((provider) => (
                <div key={provider.provider_id} className="flex flex-col items-center text-center">
                  <div className="relative w-12 h-12 rounded-lg overflow-hidden mb-2">
                    <Image
                      src={`https://image.tmdb.org/t/p/original${provider.logo_path}`}
                      alt={provider.provider_name}
                      fill
                      className="object-cover"
                    />
                  </div>
                  <span className="text-xs text-muted-foreground">{provider.provider_name}</span>
                </div>
              ))}
            </div>
          </div>
        )}

        {rent.length > 0 && (
          <div>
            <Badge variant="outline" className="mb-3">
              Rent
            </Badge>
            <div className="grid grid-cols-3 gap-3">
              {rent.map((provider) => (
                <div key={provider.provider_id} className="flex flex-col items-center text-center">
                  <div className="relative w-12 h-12 rounded-lg overflow-hidden mb-2">
                    <Image
                      src={`https://image.tmdb.org/t/p/original${provider.logo_path}`}
                      alt={provider.provider_name}
                      fill
                      className="object-cover"
                    />
                  </div>
                  <span className="text-xs text-muted-foreground">{provider.provider_name}</span>
                </div>
              ))}
            </div>
          </div>
        )}

        {buy.length > 0 && (
          <div>
            <Badge variant="outline" className="mb-3">
              Buy
            </Badge>
            <div className="grid grid-cols-3 gap-3">
              {buy.map((provider) => (
                <div key={provider.provider_id} className="flex flex-col items-center text-center">
                  <div className="relative w-12 h-12 rounded-lg overflow-hidden mb-2">
                    <Image
                      src={`https://image.tmdb.org/t/p/original${provider.logo_path}`}
                      alt={provider.provider_name}
                      fill
                      className="object-cover"
                    />
                  </div>
                  <span className="text-xs text-muted-foreground">{provider.provider_name}</span>
                </div>
              ))}
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  )
}
