"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Play, X } from "lucide-react"

interface Video {
  id: string
  key: string
  name: string
  site: string
  type: string
}

interface TVTrailerProps {
  videos: Video[]
}

export function TVTrailer({ videos }: TVTrailerProps) {
  const [selectedVideo, setSelectedVideo] = useState<Video | null>(null)

  const trailers = videos.filter(
    (video) => video.site === "YouTube" && (video.type === "Trailer" || video.type === "Teaser"),
  )

  if (trailers.length === 0) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Trailers</CardTitle>
        </CardHeader>
        <CardContent>
          <p className="text-muted-foreground">No trailers available</p>
        </CardContent>
      </Card>
    )
  }

  return (
    <>
      <Card>
        <CardHeader>
          <CardTitle>Trailers & Videos</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {trailers.slice(0, 4).map((video) => (
              <div key={video.id} className="relative group cursor-pointer" onClick={() => setSelectedVideo(video)}>
                <div className="relative aspect-video bg-muted rounded-lg overflow-hidden">
                  <img
                    src={`https://img.youtube.com/vi/${video.key}/hqdefault.jpg`}
                    alt={video.name}
                    className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-200"
                  />
                  <div className="absolute inset-0 bg-black/40 group-hover:bg-black/20 transition-colors duration-200" />
                  <div className="absolute inset-0 flex items-center justify-center">
                    <Button size="icon" variant="secondary" className="rounded-full">
                      <Play className="h-6 w-6 fill-current" />
                    </Button>
                  </div>
                </div>
                <h4 className="mt-2 text-sm font-medium line-clamp-2">{video.name}</h4>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Video Modal */}
      {selectedVideo && (
        <div className="fixed inset-0 z-50 bg-black/80 flex items-center justify-center p-4">
          <div className="relative w-full max-w-4xl aspect-video">
            <Button
              variant="ghost"
              size="icon"
              className="absolute -top-12 right-0 text-white hover:bg-white/20"
              onClick={() => setSelectedVideo(null)}
            >
              <X className="h-6 w-6" />
            </Button>
            <iframe
              src={`https://www.youtube.com/embed/${selectedVideo.key}?autoplay=1`}
              title={selectedVideo.name}
              className="w-full h-full rounded-lg"
              allowFullScreen
              allow="autoplay; encrypted-media"
            />
          </div>
        </div>
      )}
    </>
  )
}
