import Image from "next/image"
import Link from "next/link"
import { Badge } from "@/components/ui/badge"
import { Card, CardContent } from "@/components/ui/card"
import { Star, Calendar, Clock, Users, Tv } from "lucide-react"
import { Button } from "@/components/ui/button"

interface TVDetailsProps {
  tvShow: {
    id: number
    name: string
    overview: string
    poster_path: string | null
    backdrop_path: string | null
    first_air_date: string
    last_air_date: string
    number_of_seasons: number
    number_of_episodes: number
    episode_run_time: number[]
    vote_average: number
    vote_count: number
    status: string
    genres: Array<{ id: number; name: string }>
    created_by: Array<{ id: number; name: string }>
    networks: Array<{ id: number; name: string; logo_path: string }>
    credits?: {
      cast: Array<{
        id: number
        name: string
        character: string
        profile_path: string | null
      }>
      crew: Array<{
        id: number
        name: string
        job: string
      }>
    }
  }
}

export function TVDetails({ tvShow }: TVDetailsProps) {
  const creators = tvShow.created_by || []
  const mainCast = tvShow.credits?.cast.slice(0, 6) || []
  const averageRuntime = tvShow.episode_run_time.length > 0 ? tvShow.episode_run_time[0] : 0

  return (
    <div className="relative">
      {/* Backdrop */}
      {tvShow.backdrop_path && (
        <div className="absolute inset-0 -z-10">
          <Image
            src={`https://image.tmdb.org/t/p/original${tvShow.backdrop_path}`}
            alt={tvShow.name}
            fill
            className="object-cover opacity-20"
          />
          <div className="absolute inset-0 bg-gradient-to-t from-background via-background/80 to-background/40" />
        </div>
      )}

      <div className="relative z-10 grid grid-cols-1 lg:grid-cols-3 gap-8 py-8">
        {/* Poster */}
        <div className="lg:col-span-1">
          <Card className="overflow-hidden">
            <CardContent className="p-0">
              <div className="relative aspect-[2/3]">
                {tvShow.poster_path ? (
                  <Image
                    src={`https://image.tmdb.org/t/p/w500${tvShow.poster_path}`}
                    alt={tvShow.name}
                    fill
                    className="object-cover"
                  />
                ) : (
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <span className="text-muted-foreground">No Image</span>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Details */}
        <div className="lg:col-span-2 space-y-6">
          <div>
            <h1 className="text-4xl font-bold mb-4">{tvShow.name}</h1>

            <div className="flex flex-wrap items-center gap-4 mb-6">
              <Badge variant="secondary" className="text-sm">
                <Star className="w-4 h-4 mr-1 fill-yellow-400 text-yellow-400" />
                {tvShow.vote_average.toFixed(1)} ({tvShow.vote_count.toLocaleString()} votes)
              </Badge>

              {tvShow.first_air_date && (
                <Badge variant="outline" className="text-sm">
                  <Calendar className="w-4 h-4 mr-1" />
                  {new Date(tvShow.first_air_date).getFullYear()}
                  {tvShow.last_air_date &&
                    tvShow.status === "Ended" &&
                    ` - ${new Date(tvShow.last_air_date).getFullYear()}`}
                </Badge>
              )}

              <Badge variant="outline" className="text-sm">
                <Tv className="w-4 h-4 mr-1" />
                {tvShow.number_of_seasons} Season{tvShow.number_of_seasons !== 1 ? "s" : ""}
              </Badge>

              <Badge variant="outline" className="text-sm">
                <Users className="w-4 h-4 mr-1" />
                {tvShow.number_of_episodes} Episodes
              </Badge>

              {averageRuntime > 0 && (
                <Badge variant="outline" className="text-sm">
                  <Clock className="w-4 h-4 mr-1" />
                  {averageRuntime}m per episode
                </Badge>
              )}
            </div>

            <div className="flex flex-wrap gap-2 mb-6">
              {tvShow.genres.map((genre) => (
                <Badge key={genre.id} variant="secondary">
                  {genre.name}
                </Badge>
              ))}
            </div>

            <div className="mb-6">
              <Badge variant={tvShow.status === "Returning Series" ? "default" : "secondary"}>{tvShow.status}</Badge>
            </div>

            <p className="text-lg text-muted-foreground leading-relaxed mb-6">{tvShow.overview}</p>

            {creators.length > 0 && (
              <div className="mb-6">
                <h3 className="text-lg font-semibold mb-2">Created By</h3>
                <p className="text-muted-foreground">{creators.map((creator) => creator.name).join(", ")}</p>
              </div>
            )}

            {tvShow.networks && tvShow.networks.length > 0 && (
              <div className="mb-6">
                <h3 className="text-lg font-semibold mb-4">Networks</h3>
                <div className="flex flex-wrap gap-4">
                  {tvShow.networks.map((network) => (
                    <div key={network.id} className="flex items-center space-x-2">
                      {network.logo_path ? (
                        <div className="relative w-12 h-8">
                          <Image
                            src={`https://image.tmdb.org/t/p/w92${network.logo_path}`}
                            alt={network.name}
                            fill
                            className="object-contain"
                          />
                        </div>
                      ) : (
                        <span className="text-sm font-medium">{network.name}</span>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}

            {mainCast.length > 0 && (
              <div>
                <h3 className="text-lg font-semibold mb-4">Cast</h3>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  {mainCast.map((actor) => (
                    <Link key={actor.id} href={`/person/${actor.id}`}>
                      <div className="flex items-center space-x-3 hover:bg-muted/50 p-2 rounded-lg transition-colors cursor-pointer">
                        <div className="relative w-12 h-12 rounded-full overflow-hidden bg-muted">
                          {actor.profile_path ? (
                            <Image
                              src={`https://image.tmdb.org/t/p/w185${actor.profile_path}`}
                              alt={actor.name}
                              fill
                              className="object-cover"
                            />
                          ) : (
                            <div className="w-full h-full flex items-center justify-center">
                              <span className="text-xs text-muted-foreground">N/A</span>
                            </div>
                          )}
                        </div>
                        <div className="min-w-0 flex-1">
                          <p className="text-sm font-medium truncate hover:text-primary">{actor.name}</p>
                          <p className="text-xs text-muted-foreground truncate">{actor.character}</p>
                        </div>
                      </div>
                    </Link>
                  ))}
                </div>
              </div>
            )}

            {tvShow.number_of_seasons > 0 && (
              <div className="mt-6">
                <Button asChild size="lg">
                  <Link href={`/tv/${tvShow.id}/seasons`}>
                    <Tv className="w-4 h-4 mr-2" />
                    View All Seasons ({tvShow.number_of_seasons})
                  </Link>
                </Button>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
