"use client"

import { useState } from "react"
import Image from "next/image"
import Link from "next/link"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Star, Calendar, Film, Tv } from "lucide-react"

interface Credit {
  id: number
  title?: string
  name?: string
  poster_path: string | null
  release_date?: string
  first_air_date?: string
  vote_average: number
  character?: string
  job?: string
  department?: string
}

interface PersonCreditsProps {
  movieCredits: {
    cast: Credit[]
    crew: Credit[]
  }
  tvCredits: {
    cast: Credit[]
    crew: Credit[]
  }
}

export function PersonCredits({ movieCredits, tvCredits }: PersonCreditsProps) {
  const [showAllMovies, setShowAllMovies] = useState(false)
  const [showAllTV, setShowAllTV] = useState(false)

  // Sort credits by date (newest first)
  const sortByDate = (credits: Credit[]) => {
    return credits.sort((a, b) => {
      const dateA = new Date(a.release_date || a.first_air_date || "1900-01-01")
      const dateB = new Date(b.release_date || b.first_air_date || "1900-01-01")
      return dateB.getTime() - dateA.getTime()
    })
  }

  const movieCast = sortByDate(movieCredits.cast || [])
  const movieCrew = sortByDate(movieCredits.crew || [])
  const tvCast = sortByDate(tvCredits.cast || [])
  const tvCrew = sortByDate(tvCredits.crew || [])

  const displayedMovies = showAllMovies ? movieCast : movieCast.slice(0, 12)
  const displayedTV = showAllTV ? tvCast : tvCast.slice(0, 12)

  const CreditCard = ({ credit, type }: { credit: Credit; type: "movie" | "tv" }) => {
    const title = credit.title || credit.name || "Unknown Title"
    const date = credit.release_date || credit.first_air_date
    const year = date ? new Date(date).getFullYear() : null
    const linkHref = type === "movie" ? `/movie/${credit.id}` : `/tv/${credit.id}`

    return (
      <Link href={linkHref}>
        <Card className="group hover:shadow-lg transition-shadow duration-200">
          <CardContent className="p-0">
            <div className="relative aspect-[2/3] overflow-hidden rounded-t-lg">
              {credit.poster_path ? (
                <Image
                  src={`https://image.tmdb.org/t/p/w300${credit.poster_path}`}
                  alt={title}
                  fill
                  className="object-cover group-hover:scale-105 transition-transform duration-200"
                />
              ) : (
                <div className="w-full h-full bg-muted flex items-center justify-center">
                  {type === "movie" ? (
                    <Film className="h-8 w-8 text-muted-foreground" />
                  ) : (
                    <Tv className="h-8 w-8 text-muted-foreground" />
                  )}
                </div>
              )}

              {credit.vote_average > 0 && (
                <div className="absolute top-2 right-2">
                  <Badge variant="secondary" className="text-xs">
                    <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                    {credit.vote_average.toFixed(1)}
                  </Badge>
                </div>
              )}
            </div>

            <div className="p-3 space-y-2">
              <h3 className="font-semibold text-sm line-clamp-2 group-hover:text-primary">{title}</h3>

              {year && (
                <div className="flex items-center text-xs text-muted-foreground">
                  <Calendar className="w-3 h-3 mr-1" />
                  {year}
                </div>
              )}

              {credit.character && <p className="text-xs text-muted-foreground line-clamp-1">as {credit.character}</p>}

              {credit.job && <p className="text-xs text-muted-foreground line-clamp-1">{credit.job}</p>}
            </div>
          </CardContent>
        </Card>
      </Link>
    )
  }

  return (
    <Tabs defaultValue="movies" className="space-y-6">
      <TabsList className="grid w-full grid-cols-2">
        <TabsTrigger value="movies" className="flex items-center gap-2">
          <Film className="w-4 h-4" />
          Movies ({movieCast.length})
        </TabsTrigger>
        <TabsTrigger value="tv" className="flex items-center gap-2">
          <Tv className="w-4 h-4" />
          TV Shows ({tvCast.length})
        </TabsTrigger>
      </TabsList>

      <TabsContent value="movies" className="space-y-6">
        {movieCast.length > 0 ? (
          <>
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
              {displayedMovies.map((credit) => (
                <CreditCard key={`movie-${credit.id}-${credit.character}`} credit={credit} type="movie" />
              ))}
            </div>

            {movieCast.length > 12 && (
              <div className="text-center">
                <Button variant="outline" onClick={() => setShowAllMovies(!showAllMovies)}>
                  {showAllMovies ? "Show Less" : `Show All ${movieCast.length} Movies`}
                </Button>
              </div>
            )}
          </>
        ) : (
          <div className="text-center py-12">
            <Film className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <p className="text-muted-foreground">No movie credits available.</p>
          </div>
        )}

        {movieCrew.length > 0 && (
          <div className="mt-8">
            <h3 className="text-xl font-semibold mb-4">Behind the Camera</h3>
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
              {movieCrew.slice(0, 12).map((credit) => (
                <CreditCard key={`movie-crew-${credit.id}-${credit.job}`} credit={credit} type="movie" />
              ))}
            </div>
          </div>
        )}
      </TabsContent>

      <TabsContent value="tv" className="space-y-6">
        {tvCast.length > 0 ? (
          <>
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
              {displayedTV.map((credit) => (
                <CreditCard key={`tv-${credit.id}-${credit.character}`} credit={credit} type="tv" />
              ))}
            </div>

            {tvCast.length > 12 && (
              <div className="text-center">
                <Button variant="outline" onClick={() => setShowAllTV(!showAllTV)}>
                  {showAllTV ? "Show Less" : `Show All ${tvCast.length} TV Shows`}
                </Button>
              </div>
            )}
          </>
        ) : (
          <div className="text-center py-12">
            <Tv className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <p className="text-muted-foreground">No TV show credits available.</p>
          </div>
        )}

        {tvCrew.length > 0 && (
          <div className="mt-8">
            <h3 className="text-xl font-semibold mb-4">Behind the Camera</h3>
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
              {tvCrew.slice(0, 12).map((credit) => (
                <CreditCard key={`tv-crew-${credit.id}-${credit.job}`} credit={credit} type="tv" />
              ))}
            </div>
          </div>
        )}
      </TabsContent>
    </Tabs>
  )
}
