import Image from "next/image"
import Link from "next/link"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Star, Calendar, Shield } from "lucide-react"

interface Movie {
  id: number
  title: string
  poster_path: string | null
  release_date: string
  vote_average: number
  overview: string
  adult: boolean
}

interface MovieGridProps {
  search?: string
  type?: "trending" | "popular" | "top_rated" | "discover"
  genre?: string
  year?: string
  sort?: string
  page?: number
  showAdultContent?: boolean
}

async function getMovies({
  search,
  type = "popular",
  genre,
  year,
  sort = "popularity.desc",
  page = 1,
  showAdultContent = false,
}: MovieGridProps) {
  const baseUrl = "https://api.themoviedb.org/3"
  let url = ""

  if (search) {
    url = `${baseUrl}/search/movie?api_key=${process.env.TMDB_API_KEY}&query=${encodeURIComponent(search)}&page=${page}&include_adult=${showAdultContent}`
  } else if (type === "discover" || genre || year || sort !== "popularity.desc") {
    // Use discover endpoint for filtering
    url = `${baseUrl}/discover/movie?api_key=${process.env.TMDB_API_KEY}&page=${page}&sort_by=${sort}&include_adult=${showAdultContent}`

    if (genre && genre !== "All Genres") {
      // Map genre name to ID
      const genreMap: { [key: string]: number } = {
        Action: 28,
        Adventure: 12,
        Animation: 16,
        Comedy: 35,
        Crime: 80,
        Documentary: 99,
        Drama: 18,
        Family: 10751,
        Fantasy: 14,
        History: 36,
        Horror: 27,
        Music: 10402,
        Mystery: 9648,
        Romance: 10749,
        "Science Fiction": 878,
        Thriller: 53,
        War: 10752,
        Western: 37,
      }
      const genreId = genreMap[genre]
      if (genreId) {
        url += `&with_genres=${genreId}`
      }
    }

    if (year && year !== "All Years") {
      url += `&year=${year}`
    }
  } else {
    // Use existing endpoints for trending, popular, top_rated
    switch (type) {
      case "trending":
        url = `${baseUrl}/trending/movie/week?api_key=${process.env.TMDB_API_KEY}&page=${page}`
        break
      case "top_rated":
        url = `${baseUrl}/movie/top_rated?api_key=${process.env.TMDB_API_KEY}&page=${page}&include_adult=${showAdultContent}`
        break
      default:
        url = `${baseUrl}/movie/popular?api_key=${process.env.TMDB_API_KEY}&page=${page}&include_adult=${showAdultContent}`
    }
  }

  try {
    const res = await fetch(url, { next: { revalidate: 3600 } })

    if (!res.ok) {
      console.error(`TMDB request failed (${res.status}) for ${url}`)
      return { results: [] }
    }

    const data = await res.json()

    // Filter adult content on server side as additional safety
    if (!showAdultContent && data.results) {
      data.results = data.results.filter((movie: Movie) => !movie.adult)
    }

    return data
  } catch (err) {
    console.error("TMDB fetch error:", err)
    return { results: [] }
  }
}

export async function MovieGrid(props: MovieGridProps) {
  const data = await getMovies(props)
  const movies: Movie[] = data.results || []

  if (movies.length === 0) {
    return (
      <div className="text-center py-12">
        <p className="text-muted-foreground">No movies found.</p>
      </div>
    )
  }

  return (
    <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-6">
      {movies.map((movie) => (
        <Link key={movie.id} href={`/movie/${movie.id}`}>
          <Card className="group hover:shadow-lg transition-shadow duration-200">
            <CardContent className="p-0">
              <div className="relative aspect-[2/3] overflow-hidden rounded-t-lg">
                {movie.poster_path ? (
                  <Image
                    src={`https://image.tmdb.org/t/p/w500${movie.poster_path}`}
                    alt={movie.title}
                    fill
                    className="object-cover group-hover:scale-105 transition-transform duration-200"
                  />
                ) : (
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <span className="text-muted-foreground">No Image</span>
                  </div>
                )}

                <div className="absolute top-2 right-2 flex flex-col gap-1">
                  <Badge variant="secondary" className="text-xs">
                    <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                    {movie.vote_average.toFixed(1)}
                  </Badge>

                  {movie.adult && (
                    <Badge variant="destructive" className="text-xs">
                      <Shield className="w-3 h-3 mr-1" />
                      18+
                    </Badge>
                  )}
                </div>
              </div>

              <div className="p-3 space-y-2">
                <h3 className="font-semibold text-sm line-clamp-2 group-hover:text-primary">{movie.title}</h3>

                {movie.release_date && (
                  <div className="flex items-center text-xs text-muted-foreground">
                    <Calendar className="w-3 h-3 mr-1" />
                    {new Date(movie.release_date).getFullYear()}
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </Link>
      ))}
    </div>
  )
}
