import Image from "next/image"
import Link from "next/link"
import { Badge } from "@/components/ui/badge"
import { Card, CardContent } from "@/components/ui/card"
import { Star, Calendar, Clock } from "lucide-react"

interface MovieDetailsProps {
  movie: {
    id: number
    title: string
    overview: string
    poster_path: string | null
    backdrop_path: string | null
    release_date: string
    runtime: number
    vote_average: number
    vote_count: number
    genres: Array<{ id: number; name: string }>
    credits?: {
      cast: Array<{
        id: number
        name: string
        character: string
        profile_path: string | null
      }>
      crew: Array<{
        id: number
        name: string
        job: string
      }>
    }
  }
}

export function MovieDetails({ movie }: MovieDetailsProps) {
  const director = movie.credits?.crew.find((person) => person.job === "Director")
  const mainCast = movie.credits?.cast.slice(0, 6) || []

  return (
    <div className="relative">
      {/* Backdrop */}
      {movie.backdrop_path && (
        <div className="absolute inset-0 -z-10">
          <Image
            src={`https://image.tmdb.org/t/p/original${movie.backdrop_path}`}
            alt={movie.title}
            fill
            className="object-cover opacity-20"
          />
          <div className="absolute inset-0 bg-gradient-to-t from-background via-background/80 to-background/40" />
        </div>
      )}

      <div className="relative z-10 grid grid-cols-1 lg:grid-cols-3 gap-8 py-8">
        {/* Poster */}
        <div className="lg:col-span-1">
          <Card className="overflow-hidden">
            <CardContent className="p-0">
              <div className="relative aspect-[2/3]">
                {movie.poster_path ? (
                  <Image
                    src={`https://image.tmdb.org/t/p/w500${movie.poster_path}`}
                    alt={movie.title}
                    fill
                    className="object-cover"
                  />
                ) : (
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <span className="text-muted-foreground">No Image</span>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Details */}
        <div className="lg:col-span-2 space-y-6">
          <div>
            <h1 className="text-4xl font-bold mb-4">{movie.title}</h1>

            <div className="flex flex-wrap items-center gap-4 mb-6">
              <Badge variant="secondary" className="text-sm">
                <Star className="w-4 h-4 mr-1 fill-yellow-400 text-yellow-400" />
                {movie.vote_average.toFixed(1)} ({movie.vote_count.toLocaleString()} votes)
              </Badge>

              {movie.release_date && (
                <Badge variant="outline" className="text-sm">
                  <Calendar className="w-4 h-4 mr-1" />
                  {new Date(movie.release_date).getFullYear()}
                </Badge>
              )}

              {movie.runtime && (
                <Badge variant="outline" className="text-sm">
                  <Clock className="w-4 h-4 mr-1" />
                  {Math.floor(movie.runtime / 60)}h {movie.runtime % 60}m
                </Badge>
              )}
            </div>

            <div className="flex flex-wrap gap-2 mb-6">
              {movie.genres.map((genre) => (
                <Badge key={genre.id} variant="secondary">
                  {genre.name}
                </Badge>
              ))}
            </div>

            <p className="text-lg text-muted-foreground leading-relaxed mb-6">{movie.overview}</p>

            {director && (
              <div className="mb-6">
                <h3 className="text-lg font-semibold mb-2">Director</h3>
                <p className="text-muted-foreground">{director.name}</p>
              </div>
            )}

            {mainCast.length > 0 && (
              <div>
                <h3 className="text-lg font-semibold mb-4">Cast</h3>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  {mainCast.map((actor) => (
                    <Link key={actor.id} href={`/person/${actor.id}`}>
                      <div className="flex items-center space-x-3 hover:bg-muted/50 p-2 rounded-lg transition-colors cursor-pointer">
                        <div className="relative w-12 h-12 rounded-full overflow-hidden bg-muted">
                          {actor.profile_path ? (
                            <Image
                              src={`https://image.tmdb.org/t/p/w185${actor.profile_path}`}
                              alt={actor.name}
                              fill
                              className="object-cover"
                            />
                          ) : (
                            <div className="w-full h-full flex items-center justify-center">
                              <span className="text-xs text-muted-foreground">N/A</span>
                            </div>
                          )}
                        </div>
                        <div className="min-w-0 flex-1">
                          <p className="text-sm font-medium truncate hover:text-primary">{actor.name}</p>
                          <p className="text-xs text-muted-foreground truncate">{actor.character}</p>
                        </div>
                      </div>
                    </Link>
                  ))}
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
