import { NextResponse } from "next/server"
import { createRouteHandlerClient } from "@supabase/auth-helpers-nextjs"
import { cookies } from "next/headers"

export async function POST() {
  if (!process.env.TMDB_API_KEY) {
    return NextResponse.json({ error: "TMDB API key not configured" }, { status: 500 })
  }

  const supabase = createRouteHandlerClient({ cookies })

  try {
    // Fetch popular movies from TMDB
    const response = await fetch(
      `https://api.themoviedb.org/3/movie/popular?api_key=${process.env.TMDB_API_KEY}&page=1`,
    )

    if (!response.ok) {
      throw new Error("Failed to fetch from TMDB")
    }

    const data = await response.json()
    const movies = data.results || []

    // Save movies to database
    const moviesToSave = movies.map((movie: any) => ({
      tmdb_id: movie.id,
      title: movie.title,
      overview: movie.overview,
      poster_path: movie.poster_path,
      backdrop_path: movie.backdrop_path,
      release_date: movie.release_date,
      vote_average: movie.vote_average,
      vote_count: movie.vote_count,
      popularity: movie.popularity,
      adult: movie.adult,
      original_language: movie.original_language,
      original_title: movie.original_title,
      genre_ids: movie.genre_ids,
    }))

    const { error } = await supabase.from("movies").upsert(moviesToSave, { onConflict: "tmdb_id" })

    if (error) {
      throw error
    }

    return NextResponse.json({
      success: true,
      count: movies.length,
      message: `Successfully synced ${movies.length} movies`,
    })
  } catch (error) {
    console.error("Sync error:", error)
    return NextResponse.json({ error: "Failed to sync movies" }, { status: 500 })
  }
}
