import { type NextRequest, NextResponse } from "next/server"
import { createRouteHandlerClient } from "@supabase/auth-helpers-nextjs"
import { cookies } from "next/headers"

export async function GET(request: NextRequest) {
  try {
    const supabase = createRouteHandlerClient({ cookies })
    const { searchParams } = new URL(request.url)

    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "10")
    const category = searchParams.get("category")
    const status = searchParams.get("status") || "published"

    const offset = (page - 1) * limit

    let query = supabase
      .from("blog_posts")
      .select(`
        *,
        blog_post_categories!inner(
          blog_categories(*)
        )
      `)
      .eq("status", status)
      .order("published_at", { ascending: false })
      .range(offset, offset + limit - 1)

    if (category) {
      query = query.eq("blog_post_categories.blog_categories.slug", category)
    }

    const { data: posts, error } = await query

    if (error) {
      console.error("Error fetching blog posts:", error)
      return NextResponse.json({ error: "Failed to fetch blog posts" }, { status: 500 })
    }

    // Get total count for pagination
    let countQuery = supabase.from("blog_posts").select("*", { count: "exact", head: true }).eq("status", status)

    if (category) {
      countQuery = countQuery.eq("blog_post_categories.blog_categories.slug", category)
    }

    const { count } = await countQuery

    return NextResponse.json({
      posts: posts || [],
      pagination: {
        page,
        limit,
        total: count || 0,
        totalPages: Math.ceil((count || 0) / limit),
      },
    })
  } catch (error) {
    console.error("Error in blog posts API:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
