const express = require("express")
const axios = require("axios")
const router = express.Router()

const TMDB_API_KEY = process.env.TMDB_API_KEY
const TMDB_BASE_URL = "https://api.themoviedb.org/3"

// Search movies and TV shows
router.get("/search", async (req, res) => {
  try {
    const { query, type = "multi" } = req.query

    if (!query) {
      return res.status(400).json({ error: "Query parameter is required" })
    }

    const response = await axios.get(`${TMDB_BASE_URL}/search/${type}`, {
      params: {
        api_key: TMDB_API_KEY,
        query: query,
        include_adult: false,
      },
    })

    res.json(response.data)
  } catch (error) {
    console.error("TMDB search error:", error)
    res.status(500).json({ error: "Failed to search movies" })
  }
})

// Get movie details
router.get("/movie/:id", async (req, res) => {
  try {
    const { id } = req.params

    const [movieResponse, creditsResponse, videosResponse] = await Promise.all([
      axios.get(`${TMDB_BASE_URL}/movie/${id}`, {
        params: { api_key: TMDB_API_KEY },
      }),
      axios.get(`${TMDB_BASE_URL}/movie/${id}/credits`, {
        params: { api_key: TMDB_API_KEY },
      }),
      axios.get(`${TMDB_BASE_URL}/movie/${id}/videos`, {
        params: { api_key: TMDB_API_KEY },
      }),
    ])

    res.json({
      movie: movieResponse.data,
      credits: creditsResponse.data,
      videos: videosResponse.data,
    })
  } catch (error) {
    console.error("TMDB movie details error:", error)
    res.status(500).json({ error: "Failed to get movie details" })
  }
})

// Get TV show details
router.get("/tv/:id", async (req, res) => {
  try {
    const { id } = req.params

    const [tvResponse, creditsResponse, videosResponse] = await Promise.all([
      axios.get(`${TMDB_BASE_URL}/tv/${id}`, {
        params: { api_key: TMDB_API_KEY },
      }),
      axios.get(`${TMDB_BASE_URL}/tv/${id}/credits`, {
        params: { api_key: TMDB_API_KEY },
      }),
      axios.get(`${TMDB_BASE_URL}/tv/${id}/videos`, {
        params: { api_key: TMDB_API_KEY },
      }),
    ])

    res.json({
      tv: tvResponse.data,
      credits: creditsResponse.data,
      videos: videosResponse.data,
    })
  } catch (error) {
    console.error("TMDB TV details error:", error)
    res.status(500).json({ error: "Failed to get TV show details" })
  }
})

// Get popular movies
router.get("/movies/popular", async (req, res) => {
  try {
    const { page = 1 } = req.query

    const response = await axios.get(`${TMDB_BASE_URL}/movie/popular`, {
      params: {
        api_key: TMDB_API_KEY,
        page: page,
      },
    })

    res.json(response.data)
  } catch (error) {
    console.error("TMDB popular movies error:", error)
    res.status(500).json({ error: "Failed to get popular movies" })
  }
})

// Get popular TV shows
router.get("/tv/popular", async (req, res) => {
  try {
    const { page = 1 } = req.query

    const response = await axios.get(`${TMDB_BASE_URL}/tv/popular`, {
      params: {
        api_key: TMDB_API_KEY,
        page: page,
      },
    })

    res.json(response.data)
  } catch (error) {
    console.error("TMDB popular TV error:", error)
    res.status(500).json({ error: "Failed to get popular TV shows" })
  }
})

module.exports = router
