class MovieApp {
  constructor() {
    this.currentUser = null
    this.init()
  }

  init() {
    this.setupEventListeners()
    this.loadPopularContent()
    this.checkAuthStatus()
  }

  setupEventListeners() {
    // Navigation
    document.querySelectorAll(".nav-link").forEach((link) => {
      link.addEventListener("click", (e) => {
        e.preventDefault()
        const section = e.target.getAttribute("href").substring(1)
        this.showSection(section)
      })
    })

    // Search
    document.getElementById("searchBtn").addEventListener("click", () => {
      this.performSearch()
    })

    document.getElementById("searchInput").addEventListener("keypress", (e) => {
      if (e.key === "Enter") {
        this.performSearch()
      }
    })

    // Auth modals
    document.getElementById("loginBtn").addEventListener("click", () => {
      this.showAuthModal("login")
    })

    document.getElementById("signupBtn").addEventListener("click", () => {
      this.showAuthModal("signup")
    })

    document.getElementById("logoutBtn").addEventListener("click", () => {
      this.logout()
    })

    // Modal controls
    document.querySelector(".close").addEventListener("click", () => {
      this.hideAuthModal()
    })

    document.getElementById("showSignup").addEventListener("click", (e) => {
      e.preventDefault()
      this.showAuthModal("signup")
    })

    document.getElementById("showLogin").addEventListener("click", (e) => {
      e.preventDefault()
      this.showAuthModal("login")
    })

    // Auth forms
    document.querySelector("#loginForm form").addEventListener("submit", (e) => {
      e.preventDefault()
      this.login()
    })

    document.querySelector("#signupForm form").addEventListener("submit", (e) => {
      e.preventDefault()
      this.signup()
    })
  }

  showSection(sectionName) {
    // Hide all sections
    document.querySelectorAll(".section").forEach((section) => {
      section.style.display = "none"
    })

    // Show selected section
    const targetSection = document.getElementById(sectionName)
    if (targetSection) {
      targetSection.style.display = "block"
    }

    // Load content based on section
    switch (sectionName) {
      case "movies":
        this.loadMovies()
        break
      case "tv":
        this.loadTVShows()
        break
      case "blog":
        this.loadBlogPosts()
        break
    }
  }

  async loadPopularContent() {
    try {
      const [moviesResponse, tvResponse] = await Promise.all([
        fetch("/api/tmdb/movies/popular"),
        fetch("/api/tmdb/tv/popular"),
      ])

      const moviesData = await moviesResponse.json()
      const tvData = await tvResponse.json()

      this.renderContent(moviesData.results, "popularMovies")
      this.renderContent(tvData.results, "popularTV")
    } catch (error) {
      console.error("Error loading popular content:", error)
    }
  }

  async loadMovies() {
    try {
      const response = await fetch("/api/tmdb/movies/popular")
      const data = await response.json()
      this.renderContent(data.results, "moviesGrid")
    } catch (error) {
      console.error("Error loading movies:", error)
    }
  }

  async loadTVShows() {
    try {
      const response = await fetch("/api/tmdb/tv/popular")
      const data = await response.json()
      this.renderContent(data.results, "tvGrid")
    } catch (error) {
      console.error("Error loading TV shows:", error)
    }
  }

  async loadBlogPosts() {
    try {
      const response = await fetch("/api/blog/posts")
      const posts = await response.json()
      this.renderBlogPosts(posts, "blogPosts")
    } catch (error) {
      console.error("Error loading blog posts:", error)
    }
  }

  async performSearch() {
    const query = document.getElementById("searchInput").value.trim()
    if (!query) return

    try {
      const response = await fetch(`/api/tmdb/search?query=${encodeURIComponent(query)}`)
      const data = await response.json()

      this.renderContent(data.results, "searchGrid")
      this.showSection("searchResults")
    } catch (error) {
      console.error("Error performing search:", error)
    }
  }

  renderContent(items, containerId) {
    const container = document.getElementById(containerId)
    if (!container) return

    container.innerHTML = items
      .map((item) => {
        const title = item.title || item.name
        const date = item.release_date || item.first_air_date
        const posterPath = item.poster_path
          ? `https://image.tmdb.org/t/p/w300${item.poster_path}`
          : "/placeholder.svg?height=300&width=200"

        return `
                <div class="content-card" onclick="app.showDetails(${item.id}, '${item.media_type || (item.title ? "movie" : "tv")}')">
                    <img src="${posterPath}" alt="${title}" loading="lazy">
                    <div class="content-card-info">
                        <h3>${title}</h3>
                        <p>${date ? new Date(date).getFullYear() : "N/A"}</p>
                        <p>⭐ ${item.vote_average?.toFixed(1) || "N/A"}</p>
                    </div>
                </div>
            `
      })
      .join("")
  }

  renderBlogPosts(posts, containerId) {
    const container = document.getElementById(containerId)
    if (!container) return

    container.innerHTML = posts
      .map(
        (post) => `
            <div class="blog-card">
                <h3>${post.title}</h3>
                <p>${post.excerpt}</p>
                <div class="blog-meta">
                    ${new Date(post.created_at).toLocaleDateString()}
                    ${post.blog_categories ? `• ${post.blog_categories.name}` : ""}
                </div>
            </div>
        `,
      )
      .join("")
  }

  async showDetails(id, type) {
    try {
      const response = await fetch(`/api/tmdb/${type}/${id}`)
      const data = await response.json()

      // Create a simple modal for details
      const modal = document.createElement("div")
      modal.className = "modal"
      modal.style.display = "block"

      const content = type === "movie" ? data.movie : data.tv
      const title = content.title || content.name

      modal.innerHTML = `
                <div class="modal-content" style="max-width: 800px;">
                    <span class="close" onclick="this.parentElement.parentElement.remove()">&times;</span>
                    <h2>${title}</h2>
                    <div style="display: flex; gap: 2rem; margin-top: 1rem;">
                        <img src="https://image.tmdb.org/t/p/w300${content.poster_path}" 
                             alt="${title}" style="width: 200px; border-radius: 8px;">
                        <div>
                            <p><strong>Overview:</strong> ${content.overview}</p>
                            <p><strong>Rating:</strong> ⭐ ${content.vote_average?.toFixed(1)}</p>
                            <p><strong>Release:</strong> ${content.release_date || content.first_air_date}</p>
                            ${content.genres ? `<p><strong>Genres:</strong> ${content.genres.map((g) => g.name).join(", ")}</p>` : ""}
                        </div>
                    </div>
                </div>
            `

      document.body.appendChild(modal)
    } catch (error) {
      console.error("Error loading details:", error)
    }
  }

  showAuthModal(type) {
    const modal = document.getElementById("authModal")
    const loginForm = document.getElementById("loginForm")
    const signupForm = document.getElementById("signupForm")

    if (type === "login") {
      loginForm.style.display = "block"
      signupForm.style.display = "none"
    } else {
      loginForm.style.display = "none"
      signupForm.style.display = "block"
    }

    modal.style.display = "block"
  }

  hideAuthModal() {
    document.getElementById("authModal").style.display = "none"
  }

  async login() {
    const email = document.getElementById("loginEmail").value
    const password = document.getElementById("loginPassword").value

    try {
      const response = await fetch("/api/auth/login", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ email, password }),
      })

      const data = await response.json()

      if (response.ok) {
        this.currentUser = data.user
        localStorage.setItem("authToken", data.session.access_token)
        this.updateAuthUI()
        this.hideAuthModal()
        alert("Login successful!")
      } else {
        alert(data.error || "Login failed")
      }
    } catch (error) {
      console.error("Login error:", error)
      alert("Login failed")
    }
  }

  async signup() {
    const full_name = document.getElementById("signupName").value
    const email = document.getElementById("signupEmail").value
    const password = document.getElementById("signupPassword").value

    try {
      const response = await fetch("/api/auth/register", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ full_name, email, password }),
      })

      const data = await response.json()

      if (response.ok) {
        alert("Registration successful! Please check your email to verify your account.")
        this.showAuthModal("login")
      } else {
        alert(data.error || "Registration failed")
      }
    } catch (error) {
      console.error("Signup error:", error)
      alert("Registration failed")
    }
  }

  async logout() {
    try {
      await fetch("/api/auth/logout", { method: "POST" })
      this.currentUser = null
      localStorage.removeItem("authToken")
      this.updateAuthUI()
      alert("Logged out successfully")
    } catch (error) {
      console.error("Logout error:", error)
    }
  }

  checkAuthStatus() {
    const token = localStorage.getItem("authToken")
    if (token) {
      // Verify token with server
      fetch("/api/auth/profile", {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      })
        .then((response) => response.json())
        .then((data) => {
          if (data.user) {
            this.currentUser = data.user
            this.updateAuthUI()
          }
        })
        .catch(() => {
          localStorage.removeItem("authToken")
        })
    }
  }

  updateAuthUI() {
    const loginBtn = document.getElementById("loginBtn")
    const signupBtn = document.getElementById("signupBtn")
    const userMenu = document.getElementById("userMenu")
    const userName = document.getElementById("userName")

    if (this.currentUser) {
      loginBtn.style.display = "none"
      signupBtn.style.display = "none"
      userMenu.style.display = "flex"
      userName.textContent = this.currentUser.user_metadata?.full_name || this.currentUser.email
    } else {
      loginBtn.style.display = "inline-block"
      signupBtn.style.display = "inline-block"
      userMenu.style.display = "none"
    }
  }
}

// Initialize the app
const app = new MovieApp()
