import Image from "next/image"
import Link from "next/link"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Calendar, Play, Star } from "lucide-react"

interface Season {
  id: number
  name: string
  overview: string
  poster_path: string | null
  season_number: number
  episode_count: number
  air_date: string
  vote_average: number
}

interface TVSeasonsOverviewProps {
  tvShow: {
    id: number
    name: string
    seasons: Season[]
  }
}

export function TVSeasonsOverview({ tvShow }: TVSeasonsOverviewProps) {
  // Filter out season 0 (specials) and sort by season number
  const regularSeasons = tvShow.seasons
    .filter((season) => season.season_number > 0)
    .sort((a, b) => a.season_number - b.season_number)

  const specials = tvShow.seasons.find((season) => season.season_number === 0)

  return (
    <div className="space-y-6">
      {/* Regular Seasons */}
      {regularSeasons.map((season) => (
        <Card key={season.id} className="overflow-hidden hover:shadow-lg transition-shadow">
          <CardContent className="p-0">
            <div className="flex flex-col md:flex-row">
              {/* Season Poster */}
              <div className="md:w-48 flex-shrink-0">
                <div className="relative aspect-[2/3] md:aspect-[3/4]">
                  {season.poster_path ? (
                    <Image
                      src={`https://image.tmdb.org/t/p/w300${season.poster_path}`}
                      alt={season.name}
                      fill
                      className="object-cover"
                    />
                  ) : (
                    <div className="w-full h-full bg-muted flex items-center justify-center">
                      <span className="text-muted-foreground">No Image</span>
                    </div>
                  )}
                </div>
              </div>

              {/* Season Details */}
              <div className="flex-1 p-6">
                <div className="flex flex-col md:flex-row md:items-start md:justify-between mb-4">
                  <div>
                    <h3 className="text-2xl font-bold mb-2">{season.name}</h3>
                    <div className="flex flex-wrap items-center gap-3 mb-4">
                      <Badge variant="secondary">Season {season.season_number}</Badge>

                      <Badge variant="outline">
                        {season.episode_count} Episode{season.episode_count !== 1 ? "s" : ""}
                      </Badge>

                      {season.air_date && (
                        <Badge variant="outline" className="text-sm">
                          <Calendar className="w-3 h-3 mr-1" />
                          {new Date(season.air_date).getFullYear()}
                        </Badge>
                      )}

                      {season.vote_average > 0 && (
                        <Badge variant="secondary" className="text-sm">
                          <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                          {season.vote_average.toFixed(1)}
                        </Badge>
                      )}
                    </div>
                  </div>

                  <Button asChild>
                    <Link href={`/tv/${tvShow.id}/season/${season.season_number}`}>
                      <Play className="w-4 h-4 mr-2" />
                      View Episodes
                    </Link>
                  </Button>
                </div>

                {season.overview && (
                  <p className="text-muted-foreground leading-relaxed line-clamp-3">{season.overview}</p>
                )}

                {!season.overview && (
                  <p className="text-muted-foreground italic">No description available for this season.</p>
                )}
              </div>
            </div>
          </CardContent>
        </Card>
      ))}

      {/* Specials Section */}
      {specials && (
        <Card className="overflow-hidden border-dashed">
          <CardContent className="p-0">
            <div className="flex flex-col md:flex-row">
              <div className="md:w-48 flex-shrink-0">
                <div className="relative aspect-[2/3] md:aspect-[3/4]">
                  {specials.poster_path ? (
                    <Image
                      src={`https://image.tmdb.org/t/p/w300${specials.poster_path}`}
                      alt={specials.name}
                      fill
                      className="object-cover"
                    />
                  ) : (
                    <div className="w-full h-full bg-muted flex items-center justify-center">
                      <span className="text-muted-foreground">No Image</span>
                    </div>
                  )}
                </div>
              </div>

              <div className="flex-1 p-6">
                <div className="flex flex-col md:flex-row md:items-start md:justify-between mb-4">
                  <div>
                    <h3 className="text-2xl font-bold mb-2">{specials.name}</h3>
                    <div className="flex flex-wrap items-center gap-3 mb-4">
                      <Badge variant="outline">Specials</Badge>

                      <Badge variant="outline">
                        {specials.episode_count} Episode{specials.episode_count !== 1 ? "s" : ""}
                      </Badge>
                    </div>
                  </div>

                  <Button variant="outline" asChild>
                    <Link href={`/tv/${tvShow.id}/season/0`}>
                      <Play className="w-4 h-4 mr-2" />
                      View Specials
                    </Link>
                  </Button>
                </div>

                <p className="text-muted-foreground leading-relaxed">
                  {specials.overview || "Special episodes, behind-the-scenes content, and bonus material."}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {regularSeasons.length === 0 && !specials && (
        <Card>
          <CardContent className="text-center py-12">
            <p className="text-muted-foreground">No season information available.</p>
          </CardContent>
        </Card>
      )}
    </div>
  )
}
