import Image from "next/image"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Calendar, Clock, Star } from "lucide-react"

interface Episode {
  id: number
  name: string
  overview: string
  still_path: string | null
  episode_number: number
  air_date: string
  runtime: number
  vote_average: number
  vote_count: number
}

interface TVEpisodesListProps {
  episodes: Episode[]
}

export function TVEpisodesList({ episodes }: TVEpisodesListProps) {
  if (!episodes || episodes.length === 0) {
    return (
      <Card>
        <CardContent className="text-center py-12">
          <p className="text-muted-foreground">No episodes available.</p>
        </CardContent>
      </Card>
    )
  }

  return (
    <div className="space-y-4">
      {episodes.map((episode) => (
        <Card key={episode.id} className="overflow-hidden hover:shadow-md transition-shadow">
          <CardContent className="p-0">
            <div className="flex flex-col sm:flex-row">
              {/* Episode Still */}
              <div className="sm:w-48 flex-shrink-0">
                <div className="relative aspect-video">
                  {episode.still_path ? (
                    <Image
                      src={`https://image.tmdb.org/t/p/w300${episode.still_path}`}
                      alt={episode.name}
                      fill
                      className="object-cover"
                    />
                  ) : (
                    <div className="w-full h-full bg-muted flex items-center justify-center">
                      <span className="text-muted-foreground text-sm">No Image</span>
                    </div>
                  )}

                  {/* Episode Number Overlay */}
                  <div className="absolute top-2 left-2">
                    <Badge variant="secondary" className="text-xs font-bold">
                      EP {episode.episode_number}
                    </Badge>
                  </div>
                </div>
              </div>

              {/* Episode Details */}
              <div className="flex-1 p-4">
                <div className="mb-3">
                  <h3 className="text-lg font-semibold mb-2 line-clamp-1">{episode.name}</h3>

                  <div className="flex flex-wrap items-center gap-2 mb-3">
                    {episode.air_date && (
                      <Badge variant="outline" className="text-xs">
                        <Calendar className="w-3 h-3 mr-1" />
                        {new Date(episode.air_date).toLocaleDateString("en-US", {
                          month: "short",
                          day: "numeric",
                          year: "numeric",
                        })}
                      </Badge>
                    )}

                    {episode.runtime && episode.runtime > 0 && (
                      <Badge variant="outline" className="text-xs">
                        <Clock className="w-3 h-3 mr-1" />
                        {episode.runtime}m
                      </Badge>
                    )}

                    {episode.vote_average > 0 && (
                      <Badge variant="secondary" className="text-xs">
                        <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                        {episode.vote_average.toFixed(1)}
                        {episode.vote_count > 0 && (
                          <span className="ml-1 text-muted-foreground">({episode.vote_count})</span>
                        )}
                      </Badge>
                    )}
                  </div>
                </div>

                {episode.overview ? (
                  <p className="text-sm text-muted-foreground leading-relaxed line-clamp-3">{episode.overview}</p>
                ) : (
                  <p className="text-sm text-muted-foreground italic">No description available for this episode.</p>
                )}
              </div>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  )
}
