"use client"

import { useRouter, useSearchParams } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Filter, TrendingUp, Star, Calendar } from "lucide-react"

interface QuickFiltersProps {
  type: "movie" | "tv" | "home"
}

export function QuickFilters({ type }: QuickFiltersProps) {
  const router = useRouter()
  const searchParams = useSearchParams()

  const handleQuickFilter = (filterType: string, value: string, basePath = "") => {
    const params = new URLSearchParams(searchParams.toString())

    // Clear existing filters when applying quick filter
    params.delete("genre")
    params.delete("year")
    params.delete("sort")
    params.delete("page")

    if (filterType === "sort") {
      params.set("sort", value)
    } else if (filterType === "genre") {
      params.set("genre", value)
    } else if (filterType === "year") {
      params.set("year", value)
    }

    const path = basePath || (type === "movie" ? "/movies" : type === "tv" ? "/tv" : "/")
    router.push(`${path}?${params.toString()}`)
  }

  const currentSort = searchParams.get("sort") || "popularity.desc"
  const currentGenre = searchParams.get("genre") || ""

  if (type === "home") {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            Quick Browse
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div>
            <h4 className="text-sm font-medium mb-3">Movies</h4>
            <div className="flex flex-wrap gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?sort=popularity.desc")}
                className="bg-transparent"
              >
                <TrendingUp className="h-4 w-4 mr-1" />
                Popular
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?sort=vote_average.desc")}
                className="bg-transparent"
              >
                <Star className="h-4 w-4 mr-1" />
                Top Rated
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?sort=release_date.desc")}
                className="bg-transparent"
              >
                <Calendar className="h-4 w-4 mr-1" />
                Latest
              </Button>
            </div>
          </div>

          <div>
            <h4 className="text-sm font-medium mb-3">TV Shows</h4>
            <div className="flex flex-wrap gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/tv?sort=popularity.desc")}
                className="bg-transparent"
              >
                <TrendingUp className="h-4 w-4 mr-1" />
                Popular
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/tv?sort=vote_average.desc")}
                className="bg-transparent"
              >
                <Star className="h-4 w-4 mr-1" />
                Top Rated
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/tv?sort=first_air_date.desc")}
                className="bg-transparent"
              >
                <Calendar className="h-4 w-4 mr-1" />
                Latest
              </Button>
            </div>
          </div>

          <div>
            <h4 className="text-sm font-medium mb-3">Popular Genres</h4>
            <div className="flex flex-wrap gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?genre=Action")}
                className="bg-transparent"
              >
                Action
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?genre=Comedy")}
                className="bg-transparent"
              >
                Comedy
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?genre=Drama")}
                className="bg-transparent"
              >
                Drama
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/movies?genre=Horror")}
                className="bg-transparent"
              >
                Horror
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    )
  }

  const sortOptions =
    type === "movie"
      ? [
          { value: "popularity.desc", label: "Popular", icon: TrendingUp },
          { value: "vote_average.desc", label: "Top Rated", icon: Star },
          { value: "release_date.desc", label: "Latest", icon: Calendar },
        ]
      : [
          { value: "popularity.desc", label: "Popular", icon: TrendingUp },
          { value: "vote_average.desc", label: "Top Rated", icon: Star },
          { value: "first_air_date.desc", label: "Latest", icon: Calendar },
        ]

  const popularGenres =
    type === "movie"
      ? ["Action", "Comedy", "Drama", "Horror", "Thriller", "Romance"]
      : ["Drama", "Comedy", "Crime", "Action & Adventure", "Sci-Fi & Fantasy", "Animation"]

  return (
    <div className="space-y-4">
      {/* Sort Quick Filters */}
      <Card>
        <CardHeader className="pb-3">
          <CardTitle className="text-lg">Sort By</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-2">
            {sortOptions.map((option) => {
              const Icon = option.icon
              const isActive = currentSort === option.value
              return (
                <Button
                  key={option.value}
                  variant={isActive ? "default" : "outline"}
                  size="sm"
                  onClick={() => handleQuickFilter("sort", option.value)}
                  className={!isActive ? "bg-transparent" : ""}
                >
                  <Icon className="h-4 w-4 mr-1" />
                  {option.label}
                </Button>
              )
            })}
          </div>
        </CardContent>
      </Card>

      {/* Genre Quick Filters */}
      <Card>
        <CardHeader className="pb-3">
          <CardTitle className="text-lg">Popular Genres</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-2">
            {popularGenres.map((genre) => {
              const isActive = currentGenre === genre
              return (
                <Button
                  key={genre}
                  variant={isActive ? "default" : "outline"}
                  size="sm"
                  onClick={() => handleQuickFilter("genre", genre)}
                  className={!isActive ? "bg-transparent" : ""}
                >
                  {genre}
                </Button>
              )
            })}
          </div>
        </CardContent>
      </Card>

      {/* Year Quick Filters */}
      <Card>
        <CardHeader className="pb-3">
          <CardTitle className="text-lg">Release Year</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-2">
            {[2024, 2023, 2022, 2021, 2020].map((year) => (
              <Button
                key={year}
                variant="outline"
                size="sm"
                onClick={() => handleQuickFilter("year", year.toString())}
                className="bg-transparent"
              >
                {year}
              </Button>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  )
}
