"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { useToast } from "@/hooks/use-toast"
import { Loader2, Sparkles, FileText, TrendingUp } from "lucide-react"
import Image from "next/image"

interface Movie {
  id: number
  title: string
  overview: string
  poster_path: string | null
  vote_average: number
  release_date: string
}

interface BlogPostGeneratorProps {
  movies: Movie[]
  onPostGenerated?: () => void
}

export function BlogPostGenerator({ movies, onPostGenerated }: BlogPostGeneratorProps) {
  const [generating, setGenerating] = useState<number | null>(null)
  const { toast } = useToast()

  const generateBlogPost = async (movie: Movie) => {
    setGenerating(movie.id)

    try {
      const response = await fetch("/api/blog/generate", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          movieId: movie.id,
          movieTitle: movie.title,
          movieOverview: movie.overview,
          moviePoster: movie.poster_path ? `https://image.tmdb.org/t/p/w500${movie.poster_path}` : null,
          type: "trending",
        }),
      })

      const data = await response.json()

      if (data.success) {
        toast({
          title: "Success!",
          description: "Blog post generated and published successfully!",
        })
        onPostGenerated?.()
      } else {
        throw new Error(data.error || "Failed to generate blog post")
      }
    } catch (error) {
      console.error("Error generating blog post:", error)
      toast({
        title: "Error",
        description: "Failed to generate blog post. Please try again.",
        variant: "destructive",
      })
    } finally {
      setGenerating(null)
    }
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Sparkles className="h-5 w-5 text-primary" />
          AI Blog Post Generator
        </CardTitle>
        <CardDescription>Generate engaging blog posts about trending movies using ChatGPT AI</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {movies.length === 0 ? (
            <div className="text-center py-8">
              <FileText className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">No trending movies available for blog generation.</p>
            </div>
          ) : (
            movies.slice(0, 6).map((movie) => (
              <div key={movie.id} className="flex items-center space-x-4 p-4 border rounded-lg">
                <div className="w-16 h-24 bg-muted rounded flex-shrink-0">
                  {movie.poster_path && (
                    <Image
                      src={`https://image.tmdb.org/t/p/w92${movie.poster_path}`}
                      alt={movie.title}
                      width={64}
                      height={96}
                      className="w-full h-full object-cover rounded"
                    />
                  )}
                </div>
                <div className="flex-1 min-w-0">
                  <h3 className="font-semibold truncate">{movie.title}</h3>
                  <p className="text-sm text-muted-foreground">
                    {movie.release_date ? new Date(movie.release_date).getFullYear() : "N/A"}
                  </p>
                  <p className="text-sm text-muted-foreground line-clamp-2">{movie.overview}</p>
                  <div className="flex items-center space-x-2 mt-2">
                    <Badge variant="secondary">⭐ {movie.vote_average?.toFixed(1) || "N/A"}</Badge>
                    <Badge variant="outline">
                      <TrendingUp className="h-3 w-3 mr-1" />
                      Trending
                    </Badge>
                  </div>
                </div>
                <Button
                  onClick={() => generateBlogPost(movie)}
                  disabled={generating !== null}
                  className="flex-shrink-0"
                >
                  {generating === movie.id ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Generating...
                    </>
                  ) : (
                    <>
                      <Sparkles className="mr-2 h-4 w-4" />
                      Generate Post
                    </>
                  )}
                </Button>
              </div>
            ))
          )}
        </div>
      </CardContent>
    </Card>
  )
}
